//+------------------------------------------------------------------+
//|                                            Between the Bands.mq4 |
//|                                Copyright  2005, David W. Thomas |
//|                                           mailto:davidwt@usa.net |
//+------------------------------------------------------------------+
#property copyright "Copyright  2005, David W. Thomas"
#property link      "mailto:davidwt@usa.net"
/*
	this was adapted, modified, and improved from between_the_bands.mql
	by InterbankFX.
*/

//---- input parameters
extern double    Lots = 1.0;
extern int       StopLoss = 15;
extern int       TakeProfit = 6;			// this is put into the order's takeprofit, in case the expert is not running.
extern int       small_takeprofit=3;	// this allows taking a smaller profit than allowed in the order's TakeProfit.
extern bool      enter_new_trades=true;// to just manage the current orders.
extern int       start_hour=0;			// can limit trading to between this start hour.
extern int       end_hour=24;				// and this end hour, inclusive.
extern int       bb_period=20;			// Bollinger Band number of periods.
extern int       bb_deviations=2;		// Bollinger Band number of deviations.
extern int       min_bb_size=6;			// it is a suggested minimum value, if not profitable with other given parameters, it will be adjusted.
extern int       max_bb_size=20;			// maximum distance in pips of the Bollinger bands to allow trading.
extern int       slippage=2;

//---- global variables
bool hours_in_order = true;
double sllevel;
double normTakeProfit = 0;
double smallTakeProfit = 0;
double minBBSize;
double maxBBSize;
double normStoploss = 0;
double slip = 0;
datetime ordertimelen = 0;
datetime minexpiration = 0;
datetime lastbuytime =0;
datetime lastselltime =0;
string ordertag;

//+------------------------------------------------------------------+
//| expert initialization function                                   |
//+------------------------------------------------------------------+
int init()
{
	if (start_hour == end_hour)
	{
		// start and end hours are equal, so set for full 24 hour trading.
		start_hour = 0;
		end_hour = 24;
	}
	hours_in_order = start_hour < end_hour;
	ordertag = "Inside BB trade " + Symbol()+ " " + Period(); // this is used to tag the orders and only maintain the orders with this tag.

	// convert from pip settings to price settings and validate parameters:
	sllevel = MarketInfo(Symbol(), MODE_STOPLEVEL);
	if (TakeProfit >= sllevel)
		normTakeProfit = TakeProfit*Point;
	smallTakeProfit = small_takeprofit*Point;
	int min_profitable_bb_size;
	// have to maintain a minimal BB size for profit.
	if (small_takeprofit > 0)
		min_profitable_bb_size = min_bb_size + small_takeprofit + MarketInfo(Symbol(), MODE_SPREAD);
	else
		min_profitable_bb_size = min_bb_size + TakeProfit + MarketInfo(Symbol(), MODE_SPREAD);
	if (min_bb_size < min_profitable_bb_size)
		min_bb_size = min_profitable_bb_size;
	minBBSize = min_bb_size*Point;
	if (max_bb_size < min_bb_size)
		max_bb_size = min_bb_size;
	maxBBSize = max_bb_size*Point;
	if (StopLoss >= sllevel)
		normStoploss = StopLoss*Point;
	sllevel = sllevel*Point;
	slip = slippage*Point;
	// how long to keep a limit order on.  also, the expiration parameter has to be at least 30 mins.
	ordertimelen = Period()*60*2;
	if (ordertimelen < 30*60)
		minexpiration = 30*60;
	else
		minexpiration = ordertimelen + 10;  // 10 more secs, to allow the expert a chance to remove the order and its arrows.

	return(0);
}

//+------------------------------------------------------------------+
//| expert deinitialization function                                 |
//+------------------------------------------------------------------+
int deinit()
{
	return(0);
}

//+------------------------------------------------------------------+
//| expert start function                                            |
//+------------------------------------------------------------------+
int start()
{
	int buycount = 0;
	int sellcount = 0;
	// get Bollinger Band data.
	double bbu = NormalizeDouble(iBands(NULL, 0, bb_period, bb_deviations, 0, PRICE_TYPICAL, MODE_UPPER, 0), Digits);
	double bbl = NormalizeDouble(iBands(NULL, 0, bb_period, bb_deviations, 0, PRICE_TYPICAL, MODE_LOWER, 0), Digits);
	double diff = bbu - bbl;
	int idiff = diff/Point;
	Comment(ordertag,"\nBB: min=",min_bb_size," max=",max_bb_size," current=", idiff);

	// handle the current orders.
	for (int i = 0; i < OrdersTotal(); i++)
	{
		if (!OrderSelect(i, SELECT_BY_POS)) continue;
		if (OrderComment() != ordertag) continue;		 // only maintain the orders with the tag for this expert, symbol, and time frame.
		switch(OrderType())
		{
			// if made small profit, close the order.
			case OP_BUY:
				if (small_takeprofit > 0 && (Bid - OrderOpenPrice() >= smallTakeProfit))
					OrderClose(OrderTicket(), OrderLots(), Bid, slippage,	LightBlue);
				buycount++;
				break;
			case OP_SELL:
				if (small_takeprofit > 0 && (OrderOpenPrice() - Ask >= smallTakeProfit))
					OrderClose(OrderTicket(), OrderLots(), Ask, slippage,	Pink);
				sellcount++;
				break;
			// if limit order open for more than 2 periods or BB has grown large, then close it.
			case OP_BUYLIMIT:
				if (CurTime() - OrderOpenTime() > ordertimelen || diff > maxBBSize)
					DeleteOrder();
				else
					buycount++;
				break;
			case OP_SELLLIMIT:
				if (CurTime() - OrderOpenTime() > ordertimelen || diff > maxBBSize)
					DeleteOrder();
				else
					sellcount++;
				break;
		}
	}

	// check conditions for possibly initiating new order.
	if (!enter_new_trades)
		return(0);
	int curhour = TimeHour(CurTime());
	if (hours_in_order)
	{
		if (curhour > end_hour || curhour < start_hour)
		{
			Comment("Not trading because its outside of the hours.");
			return(0);
		}
	}
	else if (curhour > end_hour && curhour < start_hour)
	{
		Comment("Not trading because its outside of the hours.");
		return(0);
	}

	if (diff < minBBSize || diff > maxBBSize)
	{
		Comment(ordertag,"\nBB: min=",min_bb_size," max=",max_bb_size," current=", idiff,". Not trading.");
		return(0);
	}
	// only going to use last buy/sell times if the above loop did not detect an order already put in.
	if (buycount > 0)
		lastbuytime = 0;
	if (sellcount > 0)
		lastselltime = 0;
	if ((CurTime() - lastbuytime < 6) || (CurTime() - lastselltime < 6))
		return(0);

	// although only one order can be done within 5 secs or so, this being called
	// on every tick, will eventually trigger the other order.
	double entryprice, tp = 0.0, sl = 0.0;
	if (buycount == 0)
	{
		// setup the buy order.
		entryprice = bbl + 1*Point;
		if (normTakeProfit > 0.0)
			tp = entryprice + normTakeProfit;
		if (normStoploss > 0.0)
			sl = entryprice - normStoploss;
		if (MathAbs(entryprice - Ask) <= sllevel)
		{	// cannot put in a limit order to close to the price, so only try for a market order.
			if (MathAbs(entryprice - Ask) <= slip)
			{	// if the price is within the slippage distance, put in a market order.
				OrderSend(Symbol(), OP_BUY, Lots, entryprice, slippage, sl, tp, ordertag, 0, CurTime()+minexpiration, Blue);
				lastbuytime = CurTime();
			}
		}
		else
		{
			OrderSend(Symbol(), OP_BUYLIMIT, Lots, entryprice, slippage, sl, tp, ordertag, 0, CurTime()+minexpiration, Blue);
			lastbuytime = CurTime();
		}
	}
	if (sellcount == 0)
	{
		// setup the sell order.
		entryprice = bbu - 1*Point;
		if (normTakeProfit > 0.0)
			tp = entryprice - normTakeProfit;
		if (normStoploss > 0.0)
			sl = entryprice + normStoploss;
		if (MathAbs(entryprice - Bid) <= sllevel)
		{	// cannot put in a limit order to close to the price, so only try for a market order.
			if (MathAbs(entryprice - Bid) <= slip)
			{	// if the price is within the slippage distance, put in a market order.
				OrderSend(Symbol(), OP_SELL, Lots, entryprice, slippage, sl, tp, ordertag, 0, CurTime()+minexpiration, Red);
				lastselltime = CurTime();
			}
		}
		else
		{
			OrderSend(Symbol(), OP_SELLLIMIT, Lots, entryprice, slippage, sl, tp, ordertag, 0, CurTime()+minexpiration, Red);
			lastselltime = CurTime();
		}
	}

	return(0);
}

//+------------------------------------------------------------------+
//| function to not only delete order, but remove objects for it     |
//+------------------------------------------------------------------+
void DeleteOrder()
{
	int ticket = OrderTicket();
	OrderDelete(ticket);
	string pre = "#" + ticket + " ";
	string post;
	switch (OrderType())
	{
		case OP_BUYLIMIT:
			post = "buy limit";
			break;
		case OP_SELLLIMIT:
			post = "sell limit";
			break;
		case OP_BUYSTOP:
			post = "buy stop";
			break;
		case OP_SELLSTOP:
			post = "sell stop";
			break;
	}
	// only want to keep the arrows for orders that are filled, so remove
	// those that are not filled, since they clutter the chart.
	ObjectDelete(pre + post);
	ObjectDelete(pre + "sl");
	ObjectDelete(pre + "tp");
}

//+------------------------------------------------------------------+
